<?php
$page_title = "Portal Settings";
// config.php will include the database connection ($pdo) and necessary functions
include 'header.php'; // Includes check_login(), config.php, and starts the responsive structure

// --- Functions to manage system settings ---

/**
 * Fetches a single setting value from the system_settings table.
 * @param PDO $pdo The database connection object.
 * @param string $key The setting key.
 * @return string|null The setting value or null if not found.
 */
function get_setting($pdo, $key) {
    try {
        $stmt = $pdo->prepare("SELECT setting_value FROM system_settings WHERE setting_key = ?");
        $stmt->execute([$key]);
        return $stmt->fetchColumn();
    } catch (PDOException $e) {
        error_log("Database Error (get_setting): " . $e->getMessage());
        return null;
    }
}

/**
 * Updates or inserts a setting value into the system_settings table.
 * @param PDO $pdo The database connection object.
 * @param string $key The setting key.
 * @param string $value The setting value.
 * @return bool True on success, false on failure.
 */
function update_setting($pdo, $key, $value) {
    try {
        $stmt = $pdo->prepare("INSERT INTO system_settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = ?");
        return $stmt->execute([$key, $value, $value]);
    } catch (PDOException $e) {
        error_log("Database Error (update_setting): " . $e->getMessage());
        return false;
    }
}

// --- Form Processing Logic ---
$success_message = '';
$error_message = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        $action = $_POST['action'];

        // 1. General Settings Update (Logo, Favicon, Site Name, Check Frequency)
        if ($action === 'update_general') {
            $site_name = trim($_POST['site_name'] ?? '');
            $check_frequency = (int)($_POST['check_frequency_hours'] ?? 24);

            if ($site_name) {
                update_setting($pdo, 'site_name', $site_name);
                update_setting($pdo, 'check_frequency_hours', (string)max(1, $check_frequency)); // Ensure minimum 1 hour
                $success_message = 'General settings successfully updated.';
            } else {
                $error_message = 'Site name cannot be empty.';
            }
        }
        
        // 2. Admin Password Change
        elseif ($action === 'change_password') {
            $old_password = $_POST['old_password'] ?? '';
            $new_password = $_POST['new_password'] ?? '';
            $confirm_password = $_POST['confirm_password'] ?? '';
            $admin_id = $_SESSION['admin_id'] ?? 1; // Assuming primary admin ID is 1 or fetched from session

            if (empty($old_password) || empty($new_password) || empty($confirm_password)) {
                $error_message = 'All password fields are required.';
            } elseif ($new_password !== $confirm_password) {
                $error_message = 'New password and confirm password do not match.';
            } elseif (strlen($new_password) < 8) {
                $error_message = 'The password must be at least 8 characters long.';
            } else {
                // Fetch current password hash from database
                $stmt = $pdo->prepare("SELECT password FROM portal_admins WHERE id = ?");
                $stmt->execute([$admin_id]);
                $admin = $stmt->fetch(PDO::FETCH_ASSOC);

                if ($admin && password_verify($old_password, $admin['password'])) {
                    // Update password
                    $new_hash = password_hash($new_password, PASSWORD_DEFAULT);
                    $update_stmt = $pdo->prepare("UPDATE portal_admins SET password = ? WHERE id = ?");
                    if ($update_stmt->execute([$new_hash, $admin_id])) {
                        $success_message = 'Password successfully changed. Please log in again.';
                        // Optionally force logout after password change for security
                        // session_destroy(); 
                        // header('Location: login.php');
                        // exit;
                    } else {
                        $error_message = 'Password change failed.';
                    }
                } else {
                    $error_message = 'Current password is incorrect.';
                }
            }
        }

        // 3. File Upload (Logo/Favicon)
        elseif ($action === 'upload_file') {
            $file_type = $_POST['file_type'] ?? '';
            $upload_dir = 'uploads/'; // Ensure this directory exists and is writable
            $field_name = $file_type === 'logo' ? 'logo_file' : 'favicon_file';
            $setting_key = $file_type . '_path';

            if (!empty($_FILES[$field_name]['name'])) {
                $file = $_FILES[$field_name];
                $file_extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
                $target_filename = $file_type . '_' . time() . '.' . $file_extension;
                $target_file = $upload_dir . $target_filename;

                $allowed_extensions = $file_type === 'logo' ? ['jpg', 'jpeg', 'png', 'svg'] : ['ico', 'png'];

                if (in_array($file_extension, $allowed_extensions) && $file['error'] === UPLOAD_ERR_OK) {
                    if (move_uploaded_file($file['tmp_name'], $target_file)) {
                        // Update setting in DB
                        update_setting($pdo, $setting_key, '/' . $target_file); 
                        $success_message = strtoupper($file_type) . ' successfully uploaded and saved.';
                    } else {
                        $error_message = 'Failed to upload file. Check permissions of the uploads folder.';
                    }
                } else {
                    $error_message = 'Invalid file format or upload error. (Allowed: ' . implode(', ', $allowed_extensions) . ')';
                }
            } else {
                $error_message = 'No file selected.';
            }
        }
    }
}

// --- Fetch Current Settings ---
$current_site_name = get_setting($pdo, 'site_name') ?? 'License Management Portal';
$current_logo_path = get_setting($pdo, 'logo_path') ?? '/assets/logo.png';
$current_favicon_path = get_setting($pdo, 'favicon_path') ?? '/assets/favicon.ico';
$current_check_frequency = get_setting($pdo, 'check_frequency_hours') ?? '24';
?>

<h1 class="text-3xl font-extrabold text-gray-900 dark:text-gray-100 mb-6 sm:mb-8">Portal Settings</h1>

<?php if ($success_message): ?>
    <div class="bg-green-100 dark:bg-green-900/50 border border-green-400 dark:border-green-700 text-green-700 dark:text-green-300 px-4 py-3 rounded-xl mb-4" role="alert">
        <p class="font-bold">Success!</p>
        <p class="text-sm"><?php echo $success_message; ?></p>
    </div>
<?php endif; ?>

<?php if ($error_message): ?>
    <div class="bg-red-100 dark:bg-red-900/50 border border-red-400 dark:border-red-700 text-red-700 dark:text-red-300 px-4 py-3 rounded-xl mb-4" role="alert">
        <p class="font-bold">Error!</p>
        <p class="text-sm"><?php echo $error_message; ?></p>
    </div>
<?php endif; ?>

<div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
    
    <!-- 1. General Settings Card -->
    <div class="lg:col-span-2 bg-white dark:bg-gray-800 p-6 rounded-2xl shadow-xl border border-gray-100 dark:border-gray-700">
        <h2 class="text-2xl font-semibold text-gray-800 dark:text-gray-100 mb-6 border-b pb-3 border-gray-200 dark:border-gray-700">General Settings & Domain Check</h2>
        
        <form action="settings.php" method="POST">
            <input type="hidden" name="action" value="update_general">
            
            <!-- Site Name -->
            <div class="mb-5">
                <label for="site_name" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Portal Name (Site Name)</label>
                <input type="text" id="site_name" name="site_name" value="<?php echo htmlspecialchars($current_site_name); ?>" required
                       class="mt-1 block w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg shadow-sm focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:text-white transition duration-150">
            </div>

            <!-- Check Frequency -->
            <div class="mb-5">
                <label for="check_frequency_hours" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Domain Check Frequency (Hours)</label>
                <input type="number" id="check_frequency_hours" name="check_frequency_hours" value="<?php echo htmlspecialchars($current_check_frequency); ?>" required min="1"
                       class="mt-1 block w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg shadow-sm focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:text-white transition duration-150">
                <p class="mt-2 text-xs text-gray-500 dark:text-gray-400">Client domain licenses will be automatically checked every X hours.</p>
            </div>
            
            <button type="submit" class="w-full bg-indigo-600 hover:bg-indigo-700 text-white font-semibold py-2 px-4 rounded-lg shadow-md transition duration-200 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:ring-offset-2 dark:focus:ring-offset-gray-800">
                Save Settings
            </button>
        </form>
    </div>

    <!-- 2. Logo & Favicon Upload Card (Right Column) -->
    <div class="bg-white dark:bg-gray-800 p-6 rounded-2xl shadow-xl border border-gray-100 dark:border-gray-700 flex flex-col space-y-6">
        <h2 class="text-2xl font-semibold text-gray-800 dark:text-gray-100 border-b pb-3 border-gray-200 dark:border-gray-700">Logo & Favicon</h2>

        <!-- Logo Upload -->
        <form action="settings.php" method="POST" enctype="multipart/form-data">
            <input type="hidden" name="action" value="upload_file">
            <input type="hidden" name="file_type" value="logo">
            <div class="mb-4">
                <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Upload Portal Logo</label>
                <div class="flex items-center space-x-4">
                    <img src="<?php echo htmlspecialchars($current_logo_path); ?>" alt="Current Logo" class="h-10 w-10 object-contain rounded-lg border dark:border-gray-600 p-1">
                    <span class="text-sm text-gray-500 dark:text-gray-400 truncate"><?php echo basename($current_logo_path); ?></span>
                </div>
                <input type="file" name="logo_file" accept=".png, .jpg, .jpeg, .svg" class="mt-3 block w-full text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded-full file:border-0 file:text-sm file:font-semibold file:bg-indigo-50 file:text-indigo-700 hover:file:bg-indigo-100 dark:file:bg-indigo-900 dark:file:text-indigo-300 dark:text-gray-400"/>
                <p class="mt-1 text-xs text-gray-500 dark:text-gray-400">PNG, JPG, or SVG (Max 1MB).</p>
            </div>
            <button type="submit" class="w-full bg-green-600 hover:bg-green-700 text-white font-semibold py-2 px-4 rounded-lg shadow-md transition duration-200">
                Update Logo
            </button>
        </form>
        
        <div class="border-t border-gray-200 dark:border-gray-700 pt-4"></div>

        <!-- Favicon Upload -->
        <form action="settings.php" method="POST" enctype="multipart/form-data">
            <input type="hidden" name="action" value="upload_file">
            <input type="hidden" name="file_type" value="favicon">
            <div class="mb-4">
                <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Upload Portal Favicon</label>
                <div class="flex items-center space-x-4">
                    <img src="<?php echo htmlspecialchars($current_favicon_path); ?>" alt="Current Favicon" class="h-6 w-6 object-contain">
                    <span class="text-sm text-gray-500 dark:text-gray-400 truncate"><?php echo basename($current_favicon_path); ?></span>
                </div>
                <input type="file" name="favicon_file" accept=".ico, .png" class="mt-3 block w-full text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded-full file:border-0 file:text-sm file:font-semibold file:bg-indigo-50 file:text-indigo-700 hover:file:bg-indigo-100 dark:file:bg-indigo-900 dark:file:text-indigo-300 dark:text-gray-400"/>
                <p class="mt-1 text-xs text-gray-500 dark:text-gray-400">ICO or PNG format.</p>
            </div>
            <button type="submit" class="w-full bg-green-600 hover:bg-green-700 text-white font-semibold py-2 px-4 rounded-lg shadow-md transition duration-200">
                Update Favicon
            </button>
        </form>
    </div>
</div>

<!-- 3. Admin User Management Card (Full Width Below) -->
<div class="mt-8 bg-white dark:bg-gray-800 p-6 rounded-2xl shadow-xl border border-gray-100 dark:border-gray-700">
    <h2 class="text-2xl font-semibold text-gray-800 dark:text-gray-100 mb-6 border-b pb-3 border-gray-200 dark:border-gray-700">Admin Password Change</h2>
    
    <form action="settings.php" method="POST" class="max-w-xl">
        <input type="hidden" name="action" value="change_password">

        <div class="mb-5">
            <label for="old_password" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Current Password</label>
            <input type="password" id="old_password" name="old_password" required
                   class="mt-1 block w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg shadow-sm focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:text-white transition duration-150">
        </div>

        <div class="mb-5">
            <label for="new_password" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">New Password (Min 8 Characters)</label>
            <input type="password" id="new_password" name="new_password" required
                   class="mt-1 block w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg shadow-sm focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:text-white transition duration-150">
        </div>

        <div class="mb-6">
            <label for="confirm_password" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Confirm New Password</label>
            <input type="password" id="confirm_password" name="confirm_password" required
                   class="mt-1 block w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg shadow-sm focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:text-white transition duration-150">
        </div>

        <button type="submit" class="bg-red-600 hover:bg-red-700 text-white font-semibold py-2 px-4 rounded-lg shadow-md transition duration-200 focus:outline-none focus:ring-2 focus:ring-red-500 focus:ring-offset-2 dark:focus:ring-offset-gray-800">
            Change Password
        </button>
    </form>
</div>


<!-- Note: A separate, non-database mechanism is needed for manual check trigger, which usually involves an API endpoint and task queue.
    For simplicity, we assume the automatic check runs based on the 'check_frequency_hours' setting,
    and manual checks are triggered by an action on the Domains page (domains.php) for specific entries. -->

<?php include 'footer.php'; ?>