<?php
// Set Timezone
date_default_timezone_set('Asia/Dhaka');

// Start output buffering
ob_start();

$page_title = "License Management";
include 'header.php'; 

// --- AUTO-EXPIRE LOGIC ---
if (isset($pdo) && $pdo instanceof PDO) {
    // Auto-deactivate expired licenses
    $expire_sql = "UPDATE license_keys SET is_active = 0 WHERE is_active = 1 AND expires_at IS NOT NULL AND expires_at <= NOW()";
    $pdo->exec($expire_sql);
}

// --- CONFIGURATION ---
$licenses_per_page = 10;
$current_page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
$search_term = isset($_GET['search']) ? trim(filter_input(INPUT_GET, 'search', FILTER_UNSAFE_RAW)) : '';
$status_filter = isset($_GET['status']) ? trim(filter_input(INPUT_GET, 'status', FILTER_UNSAFE_RAW)) : '';
$new_key_id = isset($_GET['new_key_id']) ? max(0, (int)$_GET['new_key_id']) : 0; 
$offset = ($current_page - 1) * $licenses_per_page; 

// --- HELPER FUNCTIONS ---
function execute_query($pdo, $sql, $params = [], $fetch_all = false) {
    if (!isset($pdo) || !$pdo instanceof PDO) return $fetch_all ? [] : 0;
    try {
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        if (strpos(trim(strtoupper($sql)), 'SELECT COUNT') === 0) return (int)$stmt->fetchColumn();
        if ($fetch_all) return $stmt->fetchAll(PDO::FETCH_ASSOC);
        return $stmt->fetch(PDO::FETCH_ASSOC); 
    } catch (PDOException $e) { return $fetch_all ? [] : 0; }
}

function generate_key($length = 64) {
    if (function_exists('random_bytes')) return bin2hex(random_bytes($length / 2));
    return substr(str_shuffle(str_repeat('0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ', $length / 2)), 0, $length);
}

// --- ACTION HANDLERS ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    
    // 1. Create License
    if (isset($_POST['action']) && $_POST['action'] === 'create_license_modal') {
        $title = trim(filter_input(INPUT_POST, 'modal_license_title', FILTER_UNSAFE_RAW)) ?: "New License - " . date("Y-m-d H:i");
        $domain = trim(filter_input(INPUT_POST, 'modal_license_domain', FILTER_SANITIZE_URL));
        $expiry = trim(filter_input(INPUT_POST, 'modal_license_expiry', FILTER_UNSAFE_RAW));
        $expires_at = !empty($expiry) ? date("Y-m-d H:i:s", strtotime($expiry)) : null;
        
        // Modules
        $mod_ecommerce = isset($_POST['mod_ecommerce']) ? 1 : 0;
        $mod_filesystem = isset($_POST['mod_filesystem']) ? 1 : 0;
        $mod_membership = isset($_POST['mod_membership']) ? 1 : 0;

        $secret = generate_key(64); $api = generate_key(32);
        
        try {
            $pdo->beginTransaction();
            $stmt = $pdo->prepare("INSERT INTO license_keys (secret_key, api_key, title, created_at, is_active, expires_at, module_ecommerce, module_filesystem, module_membership) VALUES (?, ?, ?, NOW(), 1, ?, ?, ?, ?)");
            $stmt->execute([$secret, $api, $title, $expires_at, $mod_ecommerce, $mod_filesystem, $mod_membership]);
            $lid = $pdo->lastInsertId();
            
            if (!empty($domain)) {
                $clean_domain = strtolower(preg_replace('/^https?:\/\//', '', $domain));
                if (filter_var("http://".$clean_domain, FILTER_VALIDATE_URL)) {
                    $pdo->prepare("INSERT INTO client_domains (license_key_id, domain_name, status, activated_at) VALUES (?, ?, 'active', NOW())")->execute([$lid, $clean_domain]);
                }
            }
            $pdo->commit();
            header("Location: licenses.php?new_key_id=$lid"); exit;
        } catch (Exception $e) { $pdo->rollBack(); header("Location: licenses.php?error=creation_failed"); exit; }
    }

    // 2. Edit License
    if (isset($_POST['action']) && $_POST['action'] === 'edit_license') {
        $id = (int)$_POST['license_id'];
        $title = trim($_POST['edit_title']);
        $status = (int)$_POST['edit_status']; // 1 or 0
        $expiry = trim($_POST['edit_expiry']);
        $expires_at = !empty($expiry) ? date("Y-m-d H:i:s", strtotime($expiry)) : null;

        // Modules
        $mod_ecommerce = isset($_POST['edit_mod_ecommerce']) ? 1 : 0;
        $mod_filesystem = isset($_POST['edit_mod_filesystem']) ? 1 : 0;
        $mod_membership = isset($_POST['edit_mod_membership']) ? 1 : 0;

        $sql = "UPDATE license_keys SET title = ?, is_active = ?, expires_at = ?, module_ecommerce = ?, module_filesystem = ?, module_membership = ? WHERE id = ?";
        $pdo->prepare($sql)->execute([$title, $status, $expires_at, $mod_ecommerce, $mod_filesystem, $mod_membership, $id]);
        header("Location: licenses.php?msg=updated"); exit;
    }

    // 3. Delete License
    if (isset($_POST['action']) && $_POST['action'] === 'delete_license') {
        $id = (int)$_POST['license_id'];
        $pdo->prepare("DELETE FROM client_domains WHERE license_key_id = ?")->execute([$id]);
        $pdo->prepare("DELETE FROM license_keys WHERE id = ?")->execute([$id]);
        header("Location: licenses.php?msg=deleted"); exit;
    }
}

// --- DATA FETCHING ---
$params = [];
$where = " WHERE 1=1 ";
if (!empty($search_term)) {
    $where .= " AND (lk.title LIKE :s OR lk.api_key LIKE :s) ";
    $params[':s'] = "%$search_term%";
}
if ($status_filter === 'active') $where .= " AND lk.is_active = 1 ";
elseif ($status_filter === 'inactive') $where .= " AND lk.is_active = 0 ";

$total = execute_query($pdo, "SELECT COUNT(*) FROM license_keys lk $where", $params);
$total_pages = ceil($total / $licenses_per_page);
$current_page = max(1, min($current_page, $total_pages > 0 ? $total_pages : 1));
$offset = ($current_page - 1) * $licenses_per_page;

$licenses = execute_query($pdo, "
    SELECT lk.*, (SELECT COUNT(*) FROM client_domains WHERE license_key_id = lk.id AND status = 'active') as active_count 
    FROM license_keys lk $where ORDER BY created_at DESC LIMIT $licenses_per_page OFFSET $offset
", $params, true);

// Success Data Fetch
$creation_data = null;
if ($new_key_id > 0) $creation_data = execute_query($pdo, "SELECT * FROM license_keys WHERE id = ?", [$new_key_id]);

function get_badge($active, $expire) {
    if ($active == 0) return '<span class="px-3 py-1 text-xs font-bold rounded-full bg-red-100 text-red-700 border border-red-200">Inactive</span>';
    if ($expire && strtotime($expire) < time()) return '<span class="px-3 py-1 text-xs font-bold rounded-full bg-yellow-100 text-yellow-800 border border-yellow-200">Expired</span>';
    return '<span class="px-3 py-1 text-xs font-bold rounded-full bg-green-100 text-green-700 border border-green-200 shadow-sm"><span class="w-1.5 h-1.5 rounded-full bg-green-500 inline-block mr-1.5 mb-0.5"></span>Active</span>';
}
?>

<div class="bg-gray-50/50 dark:bg-gray-900 min-h-screen py-8 font-sans transition-colors duration-200">
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        
        <!-- Toast Notification -->
        <?php if(isset($_GET['msg'])): ?>
        <div id="toast" class="fixed top-24 right-5 z-50 flex items-center w-full max-w-xs p-4 space-x-4 text-gray-500 bg-white rounded-xl shadow-2xl dark:text-gray-400 dark:bg-gray-800 border border-emerald-100 dark:border-gray-700 animate-fade-in-down" role="alert">
            <div class="inline-flex items-center justify-center flex-shrink-0 w-8 h-8 text-emerald-500 bg-emerald-100 rounded-lg dark:bg-emerald-900/50 dark:text-emerald-300">
                <svg class="w-5 h-5" fill="currentColor" viewBox="0 0 20 20"><path fill-rule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clip-rule="evenodd"></path></svg>
            </div>
            <div class="ml-3 text-sm font-medium text-gray-800 dark:text-gray-200">Operation Successful</div>
            <button onclick="document.getElementById('toast').remove()" class="ml-auto -mx-1.5 -my-1.5 bg-white text-gray-400 hover:text-gray-900 rounded-lg focus:ring-2 focus:ring-gray-300 p-1.5 hover:bg-gray-100 inline-flex h-8 w-8 dark:text-gray-500 dark:hover:text-white dark:bg-gray-800 dark:hover:bg-gray-700">
                <svg class="w-5 h-5" fill="currentColor" viewBox="0 0 20 20"><path fill-rule="evenodd" d="M4.293 4.293a1 1 0 011.414 0L10 8.586l4.293-4.293a1 1 0 111.414 1.414L11.414 10l4.293 4.293a1 1 0 01-1.414 1.414L10 11.414l-4.293 4.293a1 1 0 01-1.414-1.414L8.586 10 4.293 5.707a1 1 0 010-1.414z" clip-rule="evenodd"></path></svg>
            </button>
        </div>
        <?php endif; ?>

        <!-- Header Section -->
        <div class="mb-10 relative overflow-hidden rounded-3xl bg-gradient-to-r from-indigo-600 to-purple-700 p-8 shadow-2xl">
            <div class="absolute top-0 right-0 -mr-16 -mt-16 w-64 h-64 rounded-full bg-white opacity-10 blur-3xl"></div>
            <div class="relative z-10 flex flex-col md:flex-row justify-between items-center text-white">
                <div>
                    <h1 class="text-3xl font-extrabold tracking-tight mb-2">License Management</h1>
                    <p class="text-indigo-100 text-lg font-medium opacity-90">Create, track, and manage software licenses.</p>
                </div>
                <div class="mt-6 md:mt-0">
                    <button onclick="document.getElementById('create-modal').classList.remove('hidden')" class="bg-white text-indigo-600 hover:bg-indigo-50 font-bold py-3 px-6 rounded-xl shadow-lg transition-transform transform hover:scale-105 flex items-center">
                        <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"></path></svg>
                        Create New License
                    </button>
                </div>
            </div>
        </div>

        <!-- Filters -->
        <div class="bg-white dark:bg-gray-800 rounded-2xl shadow-sm border border-gray-100 dark:border-gray-700 p-5 mb-8">
            <form class="flex flex-col md:flex-row gap-4 items-end">
                <div class="relative flex-grow w-full">
                    <label class="block text-xs font-bold text-gray-500 dark:text-gray-400 uppercase mb-1 ml-1">Search Database</label>
                    <div class="relative">
                        <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                            <svg class="h-5 w-5 text-gray-400" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"/></svg>
                        </div>
                        <input type="text" name="search" value="<?= htmlspecialchars($search_term) ?>" placeholder="Search by Client Name, API Key..." 
                               class="block w-full pl-10 pr-3 py-2.5 border border-gray-200 rounded-xl leading-5 bg-gray-50 dark:bg-gray-900 dark:border-gray-600 dark:text-white placeholder-gray-400 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 transition-all">
                    </div>
                </div>
                <div class="flex-shrink-0 w-full md:w-48">
                    <label class="block text-xs font-bold text-gray-500 dark:text-gray-400 uppercase mb-1 ml-1">Status</label>
                    <select name="status" class="block w-full py-2.5 px-3 border border-gray-200 rounded-xl leading-5 bg-gray-50 dark:bg-gray-900 dark:border-gray-600 dark:text-white focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500">
                        <option value="">All Statuses</option>
                        <option value="active" <?= $status_filter=='active'?'selected':'' ?>>Active</option>
                        <option value="inactive" <?= $status_filter=='inactive'?'selected':'' ?>>Inactive</option>
                    </select>
                </div>
                <button type="submit" class="w-full md:w-auto inline-flex justify-center items-center px-6 py-2.5 border border-transparent text-sm font-bold rounded-xl text-white bg-gray-800 hover:bg-gray-900 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-gray-800 transition-all shadow-md">
                    Apply Filter
                </button>
            </form>
        </div>

        <!-- Table -->
        <div class="bg-white dark:bg-gray-800 rounded-2xl shadow-xl overflow-hidden border border-gray-100 dark:border-gray-700">
            <div class="overflow-x-auto">
                <table class="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                    <thead class="bg-gray-50/50 dark:bg-gray-700/50">
                        <tr>
                            <th class="px-6 py-4 text-left text-xs font-bold text-gray-500 dark:text-gray-300 uppercase tracking-wider">License</th>
                            <th class="px-6 py-4 text-left text-xs font-bold text-gray-500 dark:text-gray-300 uppercase tracking-wider hidden lg:table-cell">Modules</th>
                            <th class="px-6 py-4 text-left text-xs font-bold text-gray-500 dark:text-gray-300 uppercase tracking-wider hidden md:table-cell">Expiry</th>
                            <th class="px-6 py-4 text-center text-xs font-bold text-gray-500 dark:text-gray-300 uppercase tracking-wider">Status</th>
                            <th class="px-6 py-4 text-right text-xs font-bold text-gray-500 dark:text-gray-300 uppercase tracking-wider">Actions</th>
                        </tr>
                    </thead>
                    <tbody class="bg-white dark:bg-gray-800 divide-y divide-gray-100 dark:divide-gray-700">
                        <?php if(count($licenses) > 0): foreach ($licenses as $l): ?>
                        <tr class="hover:bg-gray-50 dark:hover:bg-gray-700/50 transition-colors duration-150">
                            <td class="px-6 py-4 whitespace-nowrap">
                                <div class="flex items-center">
                                    <div class="flex-shrink-0 h-10 w-10 rounded-full bg-gradient-to-br from-indigo-100 to-indigo-50 dark:from-indigo-900 dark:to-indigo-800 flex items-center justify-center text-indigo-600 dark:text-indigo-300 font-bold text-lg border border-indigo-100 dark:border-indigo-700 shadow-sm">
                                        <?= strtoupper(substr($l['title'], 0, 1)) ?>
                                    </div>
                                    <div class="ml-4">
                                        <div class="text-sm font-bold text-gray-900 dark:text-white"><?= htmlspecialchars($l['title']) ?></div>
                                        <div class="text-xs text-gray-500 dark:text-gray-400 font-mono mt-0.5 tracking-tight"><?= substr($l['api_key'], 0, 20) ?>...</div>
                                    </div>
                                </div>
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap hidden lg:table-cell">
                                <div class="flex space-x-1">
                                    <span class="px-2 py-1 text-[10px] font-bold rounded border <?= $l['module_ecommerce'] ? 'bg-blue-50 text-blue-700 border-blue-100 dark:bg-blue-900/30 dark:text-blue-300 dark:border-blue-800' : 'bg-gray-50 text-gray-400 border-gray-100 dark:bg-gray-800 dark:text-gray-600 dark:border-gray-700 opacity-50' ?>" title="eCommerce">ECOM</span>
                                    <span class="px-2 py-1 text-[10px] font-bold rounded border <?= $l['module_filesystem'] ? 'bg-purple-50 text-purple-700 border-purple-100 dark:bg-purple-900/30 dark:text-purple-300 dark:border-purple-800' : 'bg-gray-50 text-gray-400 border-gray-100 dark:bg-gray-800 dark:text-gray-600 dark:border-gray-700 opacity-50' ?>" title="File System">FILE</span>
                                    <span class="px-2 py-1 text-[10px] font-bold rounded border <?= $l['module_membership'] ? 'bg-orange-50 text-orange-700 border-orange-100 dark:bg-orange-900/30 dark:text-orange-300 dark:border-orange-800' : 'bg-gray-50 text-gray-400 border-gray-100 dark:bg-gray-800 dark:text-gray-600 dark:border-gray-700 opacity-50' ?>" title="Membership">MEMB</span>
                                </div>
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap hidden md:table-cell">
                                <div class="text-sm font-medium text-gray-700 dark:text-gray-300">
                                    <?= $l['expires_at'] ? date("d M Y", strtotime($l['expires_at'])) : "Lifetime" ?>
                                </div>
                                <?php if($l['expires_at']): ?>
                                <div class="text-xs text-gray-400"><?= date("h:i A", strtotime($l['expires_at'])) ?></div>
                                <?php endif; ?>
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap text-center">
                                <?= get_badge($l['is_active'], $l['expires_at']) ?>
                                <?php if($l['active_count']>0): ?>
                                    <div class="mt-1">
                                        <span class="inline-flex items-center px-2 py-0.5 rounded text-[10px] font-medium bg-gray-100 text-gray-600 dark:bg-gray-700 dark:text-gray-300 border border-gray-200 dark:border-gray-600">
                                            Linked
                                        </span>
                                    </div>
                                <?php endif; ?>
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                                <div class="flex items-center justify-end space-x-2">
                                    <a href="license_details.php?id=<?= $l['id'] ?>" class="text-indigo-600 hover:text-indigo-900 bg-indigo-50 hover:bg-indigo-100 dark:bg-indigo-900/20 dark:hover:bg-indigo-900/40 p-2 rounded-lg transition-colors" title="View Details">
                                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"/><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"/></svg>
                                    </a>
                                    <button onclick='openEditModal(<?= json_encode($l) ?>)' class="text-amber-600 hover:text-amber-900 bg-amber-50 hover:bg-amber-100 dark:bg-amber-900/20 dark:hover:bg-amber-900/40 p-2 rounded-lg transition-colors" title="Edit">
                                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"></path></svg>
                                    </button>
                                    <form method="POST" onsubmit="return confirm('WARNING: This will permanently delete the license and all associated domains. Continue?');" class="inline">
                                        <input type="hidden" name="action" value="delete_license">
                                        <input type="hidden" name="license_id" value="<?= $l['id'] ?>">
                                        <button class="text-rose-600 hover:text-rose-900 bg-rose-50 hover:bg-rose-100 dark:bg-rose-900/20 dark:hover:bg-rose-900/40 p-2 rounded-lg transition-colors" title="Delete">
                                            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"></path></svg>
                                        </button>
                                    </form>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; else: ?>
                        <tr><td colspan="5" class="px-6 py-12 text-center text-gray-500 italic">No licenses found. Create one to get started.</td></tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
            
            <!-- Pagination -->
            <?php if ($total_pages > 1): ?>
            <div class="bg-gray-50 dark:bg-gray-700/50 px-4 py-4 flex items-center justify-between border-t border-gray-200 dark:border-gray-700 sm:px-6">
                <div class="hidden sm:flex-1 sm:flex sm:items-center sm:justify-between">
                    <div>
                        <p class="text-sm text-gray-700 dark:text-gray-300">
                            Page <span class="font-bold"><?= $current_page ?></span> of <span class="font-bold"><?= $total_pages ?></span>
                        </p>
                    </div>
                    <nav class="relative z-0 inline-flex rounded-md shadow-sm -space-x-px">
                        <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                            <a href="?page=<?= $i ?>&search=<?= urlencode($search_term) ?>&status=<?= urlencode($status_filter) ?>" class="relative inline-flex items-center px-4 py-2 border text-sm font-medium <?= $i == $current_page ? 'z-10 bg-indigo-50 border-indigo-500 text-indigo-600 dark:bg-indigo-900/50 dark:border-indigo-600 dark:text-indigo-200' : 'bg-white border-gray-300 text-gray-500 hover:bg-gray-50 dark:bg-gray-800 dark:border-gray-600 dark:text-gray-400 dark:hover:bg-gray-700' ?>"><?= $i ?></a>
                        <?php endfor; ?>
                    </nav>
                </div>
            </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- Create Modal -->
<div id="create-modal" class="fixed inset-0 z-50 hidden overflow-y-auto" role="dialog" aria-modal="true">
    <div class="flex items-end justify-center min-h-screen pt-4 px-4 pb-20 text-center sm:block sm:p-0">
        <div class="fixed inset-0 bg-slate-900/75 transition-opacity backdrop-blur-sm" onclick="document.getElementById('create-modal').classList.add('hidden')"></div>
        <div class="inline-block align-bottom bg-white dark:bg-slate-800 rounded-2xl text-left overflow-hidden shadow-2xl transform transition-all sm:my-8 sm:align-middle sm:max-w-lg sm:w-full border border-slate-200 dark:border-slate-700">
            <div class="px-8 py-8">
                <h2 class="text-2xl font-black text-slate-900 dark:text-white mb-6 tracking-tight">Create License Key</h2>
                
                <?php if ($creation_data): ?>
                    <!-- Success View -->
                    <div class="bg-emerald-50 dark:bg-emerald-900/20 p-6 rounded-xl border border-emerald-200 dark:border-emerald-800 mb-6 text-center">
                        <div class="inline-flex items-center justify-center w-12 h-12 bg-emerald-100 rounded-full mb-3 text-emerald-600">
                            <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path></svg>
                        </div>
                        <h3 class="text-lg font-bold text-emerald-800 dark:text-emerald-300 mb-4">License Generated Successfully!</h3>
                        <div class="space-y-4 text-left">
                            <div>
                                <label class="text-xs font-bold text-emerald-600 uppercase tracking-wide">API Key</label>
                                <input readonly class="w-full text-xs font-mono p-3 bg-white dark:bg-slate-900 border border-emerald-200 dark:border-emerald-800 rounded-lg text-slate-600 dark:text-slate-300" value="<?= $creation_data['api_key'] ?>">
                            </div>
                            <div>
                                <label class="text-xs font-bold text-rose-500 uppercase tracking-wide">Secret Key (Save Now)</label>
                                <input readonly class="w-full text-xs font-mono p-3 bg-rose-50 dark:bg-rose-900/20 border border-rose-200 dark:border-rose-800 rounded-lg text-rose-700 dark:text-rose-300" value="<?= $creation_data['secret_key'] ?>">
                            </div>
                        </div>
                    </div>
                    <button onclick="window.location.href='licenses.php'" class="w-full py-3 bg-slate-900 text-white rounded-xl font-bold hover:bg-black transition-colors shadow-lg">Return to List</button>
                <?php else: ?>
                    <!-- Form View -->
                    <form method="POST" class="space-y-6">
                        <input type="hidden" name="action" value="create_license_modal">
                        <div>
                            <label class="block text-xs font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wide mb-2">License Title / Client Name</label>
                            <input type="text" name="modal_license_title" placeholder="e.g. Acme Corp - Pro Plan" class="w-full px-4 py-3 rounded-xl border border-slate-200 dark:bg-slate-700 dark:border-slate-600 dark:text-white focus:ring-4 focus:ring-indigo-500/10 focus:border-indigo-500 transition-all">
                        </div>
                        
                        <div class="bg-slate-50 dark:bg-slate-700/50 p-5 rounded-xl border border-slate-100 dark:border-slate-600">
                            <label class="block text-xs font-bold text-slate-500 dark:text-slate-400 uppercase mb-3">Module Permissions</label>
                            <div class="space-y-3">
                                <label class="flex items-center space-x-3 cursor-pointer group">
                                    <input type="checkbox" name="mod_ecommerce" value="1" checked class="h-5 w-5 text-indigo-600 border-slate-300 rounded focus:ring-indigo-500 transition-all">
                                    <span class="text-sm font-medium text-slate-700 dark:text-slate-200 group-hover:text-indigo-600 transition-colors">eCommerce Shop</span>
                                </label>
                                <label class="flex items-center space-x-3 cursor-pointer group">
                                    <input type="checkbox" name="mod_filesystem" value="1" checked class="h-5 w-5 text-indigo-600 border-slate-300 rounded focus:ring-indigo-500 transition-all">
                                    <span class="text-sm font-medium text-slate-700 dark:text-slate-200 group-hover:text-indigo-600 transition-colors">File Delivery System</span>
                                </label>
                                <label class="flex items-center space-x-3 cursor-pointer group">
                                    <input type="checkbox" name="mod_membership" value="1" checked class="h-5 w-5 text-indigo-600 border-slate-300 rounded focus:ring-indigo-500 transition-all">
                                    <span class="text-sm font-medium text-slate-700 dark:text-slate-200 group-hover:text-indigo-600 transition-colors">Membership Packages</span>
                                </label>
                            </div>
                        </div>

                        <div class="grid grid-cols-2 gap-4">
                            <div>
                                <label class="block text-xs font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wide mb-2">Expiration</label>
                                <input type="datetime-local" name="modal_license_expiry" class="w-full px-4 py-3 rounded-xl border border-slate-200 dark:bg-slate-700 dark:border-slate-600 dark:text-white focus:ring-4 focus:ring-indigo-500/10 focus:border-indigo-500 transition-all">
                            </div>
                            <div>
                                <label class="block text-xs font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wide mb-2">Initial Domain</label>
                                <input type="text" name="modal_license_domain" placeholder="example.com" class="w-full px-4 py-3 rounded-xl border border-slate-200 dark:bg-slate-700 dark:border-slate-600 dark:text-white focus:ring-4 focus:ring-indigo-500/10 focus:border-indigo-500 transition-all">
                            </div>
                        </div>

                        <div class="flex gap-3 pt-2">
                            <button type="button" onclick="document.getElementById('create-modal').classList.add('hidden')" class="flex-1 px-4 py-3 bg-white border border-slate-200 text-slate-700 font-bold rounded-xl hover:bg-slate-50 transition-colors">Cancel</button>
                            <button type="submit" class="flex-1 px-4 py-3 bg-indigo-600 text-white font-bold rounded-xl hover:bg-indigo-700 shadow-lg shadow-indigo-200 transition-all transform hover:-translate-y-0.5">Generate Key</button>
                        </div>
                    </form>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Edit Modal -->
<div id="edit-modal" class="fixed inset-0 z-50 hidden overflow-y-auto" role="dialog" aria-modal="true">
    <div class="flex items-end justify-center min-h-screen pt-4 px-4 pb-20 text-center sm:block sm:p-0">
        <div class="fixed inset-0 bg-slate-900/75 transition-opacity backdrop-blur-sm" onclick="document.getElementById('edit-modal').classList.add('hidden')"></div>
        <div class="inline-block align-bottom bg-white dark:bg-slate-800 rounded-2xl text-left overflow-hidden shadow-2xl transform transition-all sm:my-8 sm:align-middle sm:max-w-md sm:w-full border border-slate-200 dark:border-slate-700">
            <div class="bg-indigo-600 px-6 py-5"><h3 class="text-lg font-bold text-white">Update License</h3></div>
            <form method="POST" class="px-6 py-6 space-y-6">
                <input type="hidden" name="action" value="edit_license">
                <input type="hidden" name="license_id" id="edit_id">
                
                <div>
                    <label class="block text-xs font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wide mb-2">Title</label>
                    <input type="text" name="edit_title" id="edit_title" class="w-full px-4 py-3 rounded-xl border border-slate-200 dark:bg-slate-700 dark:border-slate-600 dark:text-white focus:ring-4 focus:ring-indigo-500/10 focus:border-indigo-500 transition-all">
                </div>

                <div class="bg-slate-50 dark:bg-slate-700/50 p-5 rounded-xl border border-slate-100 dark:border-slate-600">
                    <label class="block text-xs font-bold text-slate-500 dark:text-slate-400 uppercase mb-3">Module Permissions</label>
                    <div class="space-y-3">
                        <label class="flex items-center space-x-3 cursor-pointer group">
                            <input type="checkbox" name="edit_mod_ecommerce" id="edit_mod_ecommerce" value="1" class="h-5 w-5 text-indigo-600 border-slate-300 rounded focus:ring-indigo-500 transition-all">
                            <span class="text-sm font-medium text-slate-700 dark:text-slate-200 group-hover:text-indigo-600 transition-colors">eCommerce Shop</span>
                        </label>
                        <label class="flex items-center space-x-3 cursor-pointer group">
                            <input type="checkbox" name="edit_mod_filesystem" id="edit_mod_filesystem" value="1" class="h-5 w-5 text-indigo-600 border-slate-300 rounded focus:ring-indigo-500 transition-all">
                            <span class="text-sm font-medium text-slate-700 dark:text-slate-200 group-hover:text-indigo-600 transition-colors">File Delivery System</span>
                        </label>
                        <label class="flex items-center space-x-3 cursor-pointer group">
                            <input type="checkbox" name="edit_mod_membership" id="edit_mod_membership" value="1" class="h-5 w-5 text-indigo-600 border-slate-300 rounded focus:ring-indigo-500 transition-all">
                            <span class="text-sm font-medium text-slate-700 dark:text-slate-200 group-hover:text-indigo-600 transition-colors">Membership Packages</span>
                        </label>
                    </div>
                </div>
                
                <div class="grid grid-cols-2 gap-4">
                    <div>
                        <label class="block text-xs font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wide mb-2">Status</label>
                        <select name="edit_status" id="edit_status" class="w-full px-4 py-3 rounded-xl border border-slate-200 dark:bg-slate-700 dark:border-slate-600 dark:text-white focus:ring-4 focus:ring-indigo-500/10 focus:border-indigo-500 transition-all">
                            <option value="1">Active</option>
                            <option value="0">Inactive</option>
                        </select>
                    </div>
                    <div>
                        <label class="block text-xs font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wide mb-2">Expiry Date</label>
                        <input type="datetime-local" name="edit_expiry" id="edit_expiry" class="w-full px-4 py-3 rounded-xl border border-slate-200 dark:bg-slate-700 dark:border-slate-600 dark:text-white focus:ring-4 focus:ring-indigo-500/10 focus:border-indigo-500 transition-all">
                    </div>
                </div>

                <div class="flex justify-end gap-3 pt-2">
                    <button type="button" onclick="document.getElementById('edit-modal').classList.add('hidden')" class="px-5 py-2.5 bg-white border border-slate-200 text-slate-700 font-bold rounded-xl hover:bg-slate-50 transition-colors">Cancel</button>
                    <button class="px-6 py-2.5 bg-indigo-600 text-white font-bold rounded-xl hover:bg-indigo-700 shadow-lg shadow-indigo-200 transition-all transform hover:-translate-y-0.5">Save Changes</button>
                </div>
            </form>
        </div>
    </div>
</div>

<style>
    @keyframes fade-in-down { 0% { opacity: 0; transform: translateY(-20px); } 100% { opacity: 1; transform: translateY(0); } }
    .animate-fade-in-down { animation: fade-in-down 0.5s ease-out; }
</style>

<script>
    <?php if($creation_data): ?>
    document.getElementById('create-modal').classList.remove('hidden');
    <?php endif; ?>

    function openEditModal(data) {
        document.getElementById('edit_id').value = data.id;
        document.getElementById('edit_title').value = data.title;
        document.getElementById('edit_status').value = data.is_active;
        
        // Modules
        document.getElementById('edit_mod_ecommerce').checked = data.module_ecommerce == 1;
        document.getElementById('edit_mod_filesystem').checked = data.module_filesystem == 1;
        document.getElementById('edit_mod_membership').checked = data.module_membership == 1;
        
        if (data.expires_at) {
            let d = new Date(data.expires_at);
            d.setMinutes(d.getMinutes() - d.getTimezoneOffset());
            document.getElementById('edit_expiry').value = d.toISOString().slice(0, 16);
        } else {
            document.getElementById('edit_expiry').value = '';
        }
        document.getElementById('edit-modal').classList.remove('hidden');
    }
</script>

<?php include 'footer.php'; ob_end_flush(); ?>