<?php
// All admin pages should include this file
require_once 'config.php';
check_login();

// --- Dynamic Settings Retrieval Logic ---
// Assume $pdo is available after config.php

/**
 * Fetches a single setting value from the system_settings table.
 * Assumes $pdo connection is available.
 * NOTE: This function is defined here to ensure settings retrieval in header.
 */
if (!function_exists('get_setting_dynamic')) {
    function get_setting_dynamic($pdo, $key, $default = '') {
        try {
            $stmt = $pdo->prepare("SELECT setting_value FROM system_settings WHERE setting_key = ?");
            $stmt->execute([$key]);
            $value = $stmt->fetchColumn();
            return $value !== false ? $value : $default;
        } catch (PDOException $e) {
            // Log error, but return default to prevent page crash
            error_log("Settings retrieval error: " . $e->getMessage());
            return $default;
        }
    }
}

// Fetch dynamic settings or use defaults
if (isset($pdo)) {
    $site_name_dynamic = get_setting_dynamic($pdo, 'site_name', SITE_NAME); // Fallback to config SITE_NAME
    $logo_path_dynamic = get_setting_dynamic($pdo, 'logo_path', '/assets/logo.png');
    $favicon_path_dynamic = get_setting_dynamic($pdo, 'favicon_path', '/assets/favicon.ico');
} else {
    // Ultimate fallback
    $site_name_dynamic = SITE_NAME;
    $logo_path_dynamic = '/assets/logo.png';
    $favicon_path_dynamic = '/assets/favicon.ico';
}

$page_title = $page_title ?? "Dashboard"; // Set default title
?>
<!DOCTYPE html>
<html lang="en" class="light"> <!-- Default to light mode -->
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <!-- Dynamic Title -->
    <title><?php echo htmlspecialchars($page_title); ?> - <?php echo htmlspecialchars($site_name_dynamic); ?></title>
    <!-- Dynamic Favicon -->
    <link rel="icon" type="image/png" href="<?php echo htmlspecialchars($favicon_path_dynamic); ?>">
    
    <!-- Tailwind CSS CDN -->
    <script src="https://cdn.tailwindcss.com"></script>
    <!-- Inter Font -->
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700;800&display=swap" rel="stylesheet">
    <style>
        /* Custom scrollbar styles for a cleaner look in both modes */
        ::-webkit-scrollbar {
            width: 8px;
        }
        ::-webkit-scrollbar-track {
            background: #f1f1f1;
            border-radius: 10px;
        }
        .dark ::-webkit-scrollbar-track {
            background: #2d3748; /* Dark track color */
        }
        ::-webkit-scrollbar-thumb {
            background: #888;
            border-radius: 10px;
        }
        .dark ::-webkit-scrollbar-thumb {
            background: #4a5568; /* Dark thumb color */
        }
        ::-webkit-scrollbar-thumb:hover {
            background: #555;
        }

        body {
            font-family: 'Inter', sans-serif;
            transition: background-color 0.3s ease;
        }
        /* Fix for desktop sidebar visibility and main content scroll */
        @media (min-width: 768px) {
            .main-content-area {
                /* 100% minus the sidebar width (w-64 = 16rem) */
                width: calc(100% - 16rem); 
                /* Push content to the right of the fixed sidebar */
                margin-left: 16rem; 
            }
        }
    </style>
    <script>
        // --- Dark Mode Logic ---
        const userTheme = localStorage.getItem('theme');
        const systemDark = window.matchMedia('(prefers-color-scheme: dark)').matches;

        // Apply theme on load
        if (userTheme === 'dark' || (!userTheme && systemDark)) {
            document.documentElement.classList.add('dark');
        } else {
            document.documentElement.classList.remove('dark');
        }

        // Toggle function called by the switch
        function toggleTheme() {
            const isDark = document.documentElement.classList.toggle('dark');
            localStorage.setItem('theme', isDark ? 'dark' : 'light');
        }

        // --- Mobile Sidebar Logic ---
        function toggleSidebar() {
            const sidebar = document.getElementById('sidebar');
            const overlay = document.getElementById('sidebar-overlay');
            sidebar.classList.toggle('-translate-x-full');
            overlay.classList.toggle('hidden');
        }
    </script>
</head>
<body class="bg-gray-100 dark:bg-gray-800 transition-colors duration-300">
    
    <!-- Overall Flex Container -->
    <div class="flex-page min-h-screen">
        
        <!-- Sidebar and Overlay (Mobile) -->
        <?php include 'sidebar.php'; ?>
        
        <!-- Main Content Wrapper -->
        <div class="flex-1 flex flex-col overflow-hidden main-content-area">
            
            <!-- Top Bar with Hamburger Menu & Dark Mode Toggle (Header) -->
            <header class="flex justify-between items-center p-4 bg-white dark:bg-gray-900 shadow-lg z-20 transition-colors duration-300">
                <div class="flex items-center">
                    <!-- Hamburger Icon for Mobile -->
                    <button class="md:hidden text-gray-500 dark:text-gray-400 hover:text-indigo-600 mr-4 p-2 rounded-lg transition duration-150 focus:outline-none" onclick="toggleSidebar()">
                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6h16M4 12h16m-7 6h7"></path></svg>
                    </button>

                    <!-- Dynamic Logo - Increased size to h-10 -->
                    <img src="<?php echo htmlspecialchars($logo_path_dynamic); ?>" alt="<?php echo htmlspecialchars($site_name_dynamic); ?> Logo" class="h-10 mr-3 object-contain">

                    <!-- Dynamic Dashboard Title - Hidden on all screens -->
                    <!-- Removed title: <h1 class="text-2xl font-extrabold text-gray-800 dark:text-gray-100 hidden sm:block transition-colors duration-300"><?php echo htmlspecialchars($page_title); ?></h1> -->
                </div>
                
                <div class="flex items-center space-x-4">
                    <!-- Dark Mode Toggle Switch -->
                    <button id="theme-toggle" onclick="toggleTheme()" class="p-2 rounded-full text-gray-500 dark:text-yellow-400 hover:bg-gray-100 dark:hover:bg-gray-700 transition duration-300 focus:outline-none" aria-label="Toggle theme">
                        <svg id="sun-icon" class="w-6 h-6 hidden dark:block" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 3v1m0 16v1m9-9h-1M4 12H3m15.364 6.364l-.707-.707M6.343 6.343l-.707-.707m12.728 0l-.707.707M6.343 17.657l-.707.707M16 12a4 4 0 11-8 0 4 4 0 018 0z"></path></svg>
                        <svg id="moon-icon" class="w-6 h-6 dark:hidden" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20.354 15.354A9 9 0 018.646 3.646 9.003 9.003 0 0012 21a9.003 9.003 0 008.354-5.646z"></path></svg>
                    </button>

                    <!-- User Info -->
                    <div class="text-sm font-semibold text-gray-700 dark:text-gray-300">
                        Welcome, <?php echo htmlspecialchars($logged_in_user ?? 'Guest'); ?>!
                    </div>
                </div>
            </header>
            
            <!-- Page Content (Scrollable area) -->
            <main class="flex-1 p-4 sm:p-6 overflow-y-auto">
                <!-- Content starts here -->