<?php
ob_start();
$page_title = "Domain Management";
include 'header.php'; 

// --- HELPER FUNCTIONS ---
function execute_query($pdo, $sql, $params = [], $fetch_all = false) {
    if (!isset($pdo) || !$pdo instanceof PDO) return $fetch_all ? [] : 0;
    try {
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        if (strpos(trim(strtoupper($sql)), 'SELECT COUNT') === 0) return (int)$stmt->fetchColumn();
        if ($fetch_all) return $stmt->fetchAll(PDO::FETCH_ASSOC);
        return $stmt->fetch(PDO::FETCH_ASSOC); 
    } catch (PDOException $e) { return $fetch_all ? [] : 0; }
}

// UI Helper for Badges
function get_domain_status_ui($status) {
    return match($status) {
        'active' => ['bg' => 'bg-emerald-100 dark:bg-emerald-900/30', 'text' => 'text-emerald-700 dark:text-emerald-400', 'label' => 'Active', 'icon' => 'M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z', 'border' => 'border-emerald-200 dark:border-emerald-800'],
        'revoked' => ['bg' => 'bg-rose-100 dark:bg-rose-900/30', 'text' => 'text-rose-700 dark:text-rose-400', 'label' => 'Revoked', 'icon' => 'M10 14l2-2m0 0l2-2m-2 2l-2-2m2 2l2 2m7-2a9 9 0 11-18 0 9 9 0 0118 0z', 'border' => 'border-rose-200 dark:border-rose-800'],
        'expired' => ['bg' => 'bg-amber-100 dark:bg-amber-900/30', 'text' => 'text-amber-700 dark:text-amber-400', 'label' => 'Expired', 'icon' => 'M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z', 'border' => 'border-amber-200 dark:border-amber-800'],
        default => ['bg' => 'bg-gray-100 dark:bg-gray-700', 'text' => 'text-gray-700 dark:text-gray-300', 'label' => ucfirst($status), 'icon' => 'M8 12h.01M12 12h.01M16 12h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z', 'border' => 'border-gray-200 dark:border-gray-600']
    };
}

// --- ACTIONS ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // 1. Delete Domain
    if ($_POST['action'] === 'delete_domain') {
        $pdo->prepare("DELETE FROM client_domains WHERE id = ?")->execute([(int)$_POST['domain_id']]);
        header("Location: domains.php?msg=deleted"); exit;
    }
    
    // 2. Revoke Domain
    if ($_POST['action'] === 'revoke_domain') {
        $pdo->prepare("UPDATE client_domains SET status = 'revoked' WHERE id = ?")->execute([(int)$_POST['domain_id']]);
        header("Location: domains.php?msg=revoked"); exit;
    }

    // 3. Activate Domain (Strict Limit)
    if ($_POST['action'] === 'activate_domain') {
        $did = (int)$_POST['domain_id'];
        $lid = (int)execute_query($pdo, "SELECT license_key_id FROM client_domains WHERE id=?", [$did])['license_key_id'];
        
        $active = (int)execute_query($pdo, "SELECT COUNT(*) FROM client_domains WHERE license_key_id=? AND status='active' AND id != ?", [$lid, $did]);
        if ($active > 0) { header("Location: domains.php?error=limit"); exit; }
        
        $pdo->prepare("UPDATE client_domains SET status = 'active' WHERE id = ?")->execute([$did]);
        header("Location: domains.php?msg=activated"); exit;
    }
}

// --- FETCH DATA ---
$search = trim($_GET['search'] ?? '');
$status = trim($_GET['status'] ?? '');
$page = max(1, (int)($_GET['page'] ?? 1));
$limit = 20; $offset = ($page-1)*$limit;

$where = " WHERE 1=1 ";
$params = [];
if($search) { $where .= " AND (cd.domain_name LIKE ? OR lk.title LIKE ?) "; $params[]="%$search%"; $params[]="%$search%"; }
if($status) { $where .= " AND cd.status = ? "; $params[]=$status; }

$total = execute_query($pdo, "SELECT COUNT(cd.id) FROM client_domains cd JOIN license_keys lk ON cd.license_key_id = lk.id $where", $params);
$pages = ceil($total / $limit);

$domains = execute_query($pdo, "
    SELECT cd.*, lk.title as license_title, lk.id as license_id 
    FROM client_domains cd JOIN license_keys lk ON cd.license_key_id = lk.id 
    $where ORDER BY cd.activated_at DESC LIMIT $limit OFFSET $offset
", $params, true);
?>

<!-- === MAIN UI === -->
<div class="bg-gray-50 dark:bg-gray-900 min-h-screen py-10 font-sans transition-colors duration-200">
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        
        <!-- Toast Notification (Success) -->
        <?php if(isset($_GET['msg'])): ?>
        <div id="toast" class="fixed top-5 right-5 z-50 flex items-center w-full max-w-xs p-4 space-x-4 text-gray-500 bg-white rounded-xl shadow-2xl dark:text-gray-400 dark:bg-gray-800 transition-all duration-500 transform translate-y-0 opacity-100 border border-green-100 dark:border-gray-700" role="alert">
            <div class="inline-flex items-center justify-center flex-shrink-0 w-8 h-8 text-green-500 bg-green-100 rounded-lg dark:bg-green-800 dark:text-green-200">
                <svg class="w-5 h-5" fill="currentColor" viewBox="0 0 20 20"><path fill-rule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clip-rule="evenodd"></path></svg>
            </div>
            <div class="ml-3 text-sm font-medium text-gray-800 dark:text-gray-200">Action Successful</div>
            <button onclick="document.getElementById('toast').remove()" type="button" class="ml-auto -mx-1.5 -my-1.5 bg-white text-gray-400 hover:text-gray-900 rounded-lg focus:ring-2 focus:ring-gray-300 p-1.5 hover:bg-gray-100 inline-flex h-8 w-8 dark:text-gray-500 dark:hover:text-white dark:bg-gray-800 dark:hover:bg-gray-700">
                <svg class="w-5 h-5" fill="currentColor" viewBox="0 0 20 20"><path fill-rule="evenodd" d="M4.293 4.293a1 1 0 011.414 0L10 8.586l4.293-4.293a1 1 0 111.414 1.414L11.414 10l4.293 4.293a1 1 0 01-1.414 1.414L10 11.414l-4.293 4.293a1 1 0 01-1.414-1.414L8.586 10 4.293 5.707a1 1 0 010-1.414z" clip-rule="evenodd"></path></svg>
            </button>
        </div>
        <?php endif; ?>

        <!-- Toast Notification (Error) -->
        <?php if(isset($_GET['error']) && $_GET['error'] == 'limit'): ?>
        <div id="error-toast" class="fixed top-5 right-5 z-50 flex items-center w-full max-w-sm p-4 mb-4 text-red-800 border-l-4 border-red-500 bg-white dark:text-red-400 dark:bg-gray-800 shadow-xl rounded-r-xl" role="alert">
            <svg class="flex-shrink-0 w-5 h-5" fill="currentColor" viewBox="0 0 20 20"><path fill-rule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clip-rule="evenodd"></path></svg>
            <div class="ml-3 text-sm font-medium">
                <span class="font-bold">Limit Reached:</span> Only 1 active domain allowed per license.
            </div>
            <button type="button" class="ml-auto -mx-1.5 -my-1.5 bg-white text-gray-400 hover:text-gray-900 rounded-lg focus:ring-2 focus:ring-gray-300 p-1.5 hover:bg-gray-100 inline-flex h-8 w-8 dark:bg-gray-800 dark:text-gray-500 dark:hover:text-white dark:hover:bg-gray-700" onclick="document.getElementById('error-toast').remove()">
                <svg class="w-5 h-5" fill="currentColor" viewBox="0 0 20 20"><path fill-rule="evenodd" d="M4.293 4.293a1 1 0 011.414 0L10 8.586l4.293-4.293a1 1 0 111.414 1.414L11.414 10l4.293 4.293a1 1 0 01-1.414 1.414L10 11.414l-4.293 4.293a1 1 0 01-1.414-1.414L8.586 10 4.293 5.707a1 1 0 010-1.414z" clip-rule="evenodd"></path></svg>
            </button>
        </div>
        <?php endif; ?>

        <!-- Header -->
        <div class="mb-8 flex flex-col md:flex-row md:items-center md:justify-between">
            <div>
                <h1 class="text-3xl font-extrabold text-gray-900 dark:text-white tracking-tight">Domain Registry</h1>
                <p class="mt-2 text-sm text-gray-500 dark:text-gray-400">Monitor and manage all authorized domains across your ecosystem.</p>
            </div>
            <div class="mt-4 md:mt-0">
                <span class="inline-flex items-center justify-center px-4 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl shadow-sm text-sm font-medium text-gray-700 dark:text-gray-200">
                    Total Domains: <span class="ml-2 font-bold text-indigo-600 dark:text-indigo-400"><?= $total ?></span>
                </span>
            </div>
        </div>

        <!-- Advanced Filter Bar -->
        <div class="bg-white dark:bg-gray-800 rounded-2xl shadow-sm border border-gray-100 dark:border-gray-700 p-5 mb-8">
            <form class="grid grid-cols-1 md:grid-cols-12 gap-4 items-end">
                <!-- Search -->
                <div class="md:col-span-8 relative">
                    <label class="block text-xs font-semibold text-gray-500 dark:text-gray-400 uppercase tracking-wide mb-1.5 ml-1">Search Database</label>
                    <div class="relative">
                        <div class="absolute inset-y-0 left-0 pl-3.5 flex items-center pointer-events-none">
                            <svg class="h-5 w-5 text-gray-400" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"/></svg>
                        </div>
                        <input type="text" name="search" value="<?= htmlspecialchars($search) ?>" placeholder="Filter by Domain Name, License ID, or Title..." 
                               class="block w-full pl-11 pr-4 py-3 border border-gray-200 rounded-xl leading-5 bg-gray-50 dark:bg-gray-900 dark:border-gray-600 dark:text-white placeholder-gray-400 focus:outline-none focus:bg-white focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 transition-all shadow-sm">
                    </div>
                </div>
                
                <!-- Status Dropdown -->
                <div class="md:col-span-2">
                    <label class="block text-xs font-semibold text-gray-500 dark:text-gray-400 uppercase tracking-wide mb-1.5 ml-1">Status</label>
                    <select name="status" class="block w-full py-3 px-4 border border-gray-200 rounded-xl leading-5 bg-gray-50 dark:bg-gray-900 dark:border-gray-600 dark:text-white focus:outline-none focus:bg-white focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 shadow-sm appearance-none">
                        <option value="">All Statuses</option>
                        <option value="active" <?= $status=='active'?'selected':'' ?>>Active Only</option>
                        <option value="revoked" <?= $status=='revoked'?'selected':'' ?>>Revoked Only</option>
                    </select>
                </div>

                <!-- Submit Button -->
                <div class="md:col-span-2">
                    <button type="submit" class="w-full inline-flex justify-center items-center py-3 px-6 border border-transparent text-sm font-bold rounded-xl text-white bg-indigo-600 hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 transition-all shadow-md transform hover:scale-[1.02]">
                        Apply Filters
                    </button>
                </div>
            </form>
        </div>

        <!-- Modern Table -->
        <div class="bg-white dark:bg-gray-800 rounded-2xl shadow-xl overflow-hidden border border-gray-100 dark:border-gray-700">
            <div class="overflow-x-auto">
                <table class="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                    <thead class="bg-gray-50/80 dark:bg-gray-700/50 backdrop-blur-sm">
                        <tr>
                            <th scope="col" class="px-6 py-4 text-left text-xs font-bold text-gray-500 dark:text-gray-300 uppercase tracking-wider">Domain Info</th>
                            <th scope="col" class="px-6 py-4 text-left text-xs font-bold text-gray-500 dark:text-gray-300 uppercase tracking-wider hidden md:table-cell">License Reference</th>
                            <th scope="col" class="px-6 py-4 text-center text-xs font-bold text-gray-500 dark:text-gray-300 uppercase tracking-wider">Status</th>
                            <th scope="col" class="px-6 py-4 text-right text-xs font-bold text-gray-500 dark:text-gray-300 uppercase tracking-wider">Controls</th>
                        </tr>
                    </thead>
                    <tbody class="divide-y divide-gray-100 dark:divide-gray-700 bg-white dark:bg-gray-800">
                        <?php if(count($domains) > 0): foreach($domains as $d): 
                            $ui = get_domain_status_ui($d['status']);
                        ?>
                        <tr class="hover:bg-gray-50 dark:hover:bg-gray-700/40 transition-colors duration-200 group">
                            <td class="px-6 py-4 whitespace-nowrap">
                                <div class="flex items-center">
                                    <div class="flex-shrink-0 h-10 w-10 rounded-full bg-indigo-50 dark:bg-indigo-900/30 flex items-center justify-center text-indigo-600 dark:text-indigo-400 border border-indigo-100 dark:border-indigo-800">
                                        <svg class="h-5 w-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3.055 11H5a2 2 0 012 2v1a2 2 0 002 2 2 2 0 012 2v2.945M8 3.935V5.5A2.5 2.5 0 0010.5 8h.5a2 2 0 012 2 2 2 0 104 0 2 2 0 012-2h1.064M15 20.488V18a2 2 0 012-2h3.064M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>
                                    </div>
                                    <div class="ml-4">
                                        <div class="text-sm font-bold text-gray-900 dark:text-white"><?= htmlspecialchars($d['domain_name']) ?></div>
                                        <div class="text-xs text-gray-500 dark:text-gray-400 mt-0.5 flex items-center">
                                            <svg class="w-3 h-3 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>
                                            <?= date("M d, Y h:i A", strtotime($d['activated_at'])) ?>
                                        </div>
                                    </div>
                                </div>
                            </td>
                            
                            <td class="px-6 py-4 whitespace-nowrap hidden md:table-cell">
                                <a href="license_details.php?id=<?= $d['license_id'] ?>" class="group/link flex items-center p-2 -ml-2 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-700/50 transition-colors">
                                    <div class="flex-shrink-0 h-8 w-8 rounded bg-gray-100 dark:bg-gray-700 flex items-center justify-center text-gray-500 dark:text-gray-400 text-xs font-bold mr-3 border border-gray-200 dark:border-gray-600">
                                        ID
                                    </div>
                                    <div>
                                        <div class="text-sm font-medium text-indigo-600 dark:text-indigo-400 group-hover/link:underline decoration-2 underline-offset-2">
                                            <?= htmlspecialchars($d['license_title']) ?>
                                        </div>
                                        <div class="text-xs text-gray-400 dark:text-gray-500">License #<?= $d['license_id'] ?></div>
                                    </div>
                                </a>
                            </td>

                            <td class="px-6 py-4 whitespace-nowrap text-center">
                                <span class="inline-flex items-center px-3 py-1 rounded-full text-xs font-semibold <?= $ui['bg'] ?> <?= $ui['text'] ?> border <?= $ui['border'] ?>">
                                    <svg class="-ml-0.5 mr-1.5 h-3 w-3" fill="currentColor" viewBox="0 0 20 20"><path fill-rule="evenodd" d="<?= $ui['icon'] ?>" clip-rule="evenodd"></path></svg>
                                    <?= $ui['label'] ?>
                                </span>
                            </td>

                            <td class="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                                <div class="flex items-center justify-end space-x-2 opacity-80 group-hover:opacity-100 transition-opacity">
                                    
                                    <!-- Delete Button -->
                                    <form method="POST" onsubmit="return confirm('WARNING: Are you sure you want to permanently delete this domain entry? This action cannot be undone.');" class="inline-block" title="Delete Permanently">
                                        <input type="hidden" name="action" value="delete_domain">
                                        <input type="hidden" name="domain_id" value="<?= $d['id'] ?>">
                                        <button class="p-2 text-gray-400 hover:text-red-600 hover:bg-red-50 dark:hover:bg-red-900/30 rounded-lg transition-colors">
                                            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"></path></svg>
                                        </button>
                                    </form>

                                    <!-- Status Toggle -->
                                    <?php if($d['status'] == 'active'): ?>
                                    <form method="POST" class="inline-block" title="Revoke Access">
                                        <input type="hidden" name="action" value="revoke_domain">
                                        <input type="hidden" name="domain_id" value="<?= $d['id'] ?>">
                                        <button class="p-2 text-gray-400 hover:text-amber-600 hover:bg-amber-50 dark:hover:bg-amber-900/30 rounded-lg transition-colors">
                                            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M18.364 18.364A9 9 0 005.636 5.636m12.728 12.728A9 9 0 015.636 5.636m12.728 12.728L5.636 5.636"></path></svg>
                                        </button>
                                    </form>
                                    <?php else: ?>
                                    <form method="POST" class="inline-block" title="Reactivate">
                                        <input type="hidden" name="action" value="activate_domain">
                                        <input type="hidden" name="domain_id" value="<?= $d['id'] ?>">
                                        <button class="p-2 text-gray-400 hover:text-green-600 hover:bg-green-50 dark:hover:bg-green-900/30 rounded-lg transition-colors">
                                            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>
                                        </button>
                                    </form>
                                    <?php endif; ?>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; else: ?>
                        <tr>
                            <td colspan="4" class="px-6 py-16 text-center">
                                <div class="flex flex-col items-center justify-center">
                                    <div class="h-16 w-16 bg-gray-100 dark:bg-gray-800 rounded-full flex items-center justify-center mb-4">
                                        <svg class="w-8 h-8 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"></path></svg>
                                    </div>
                                    <h3 class="text-lg font-medium text-gray-900 dark:text-white">No domains found</h3>
                                    <p class="mt-1 text-sm text-gray-500 dark:text-gray-400 max-w-sm">
                                        We couldn't find any domains matching your search filters. Try adjusting your search query or clear the filters.
                                    </p>
                                    <a href="domains.php" class="mt-4 px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg shadow-sm text-sm font-medium text-gray-700 dark:text-gray-200 bg-white dark:bg-gray-700 hover:bg-gray-50 dark:hover:bg-gray-600 transition-colors">
                                        Clear Filters
                                    </a>
                                </div>
                            </td>
                        </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>

            <!-- Pagination -->
            <?php if ($pages > 1): ?>
            <div class="bg-gray-50 dark:bg-gray-800/80 px-4 py-4 flex items-center justify-between border-t border-gray-200 dark:border-gray-700 sm:px-6">
                <div class="hidden sm:flex-1 sm:flex sm:items-center sm:justify-between">
                    <div>
                        <p class="text-sm text-gray-700 dark:text-gray-300">
                            Showing page <span class="font-bold text-gray-900 dark:text-white"><?= $page ?></span> of <span class="font-bold text-gray-900 dark:text-white"><?= $pages ?></span>
                        </p>
                    </div>
                    <div>
                        <nav class="relative z-0 inline-flex rounded-lg shadow-sm -space-x-px" aria-label="Pagination">
                            <?php for ($i = 1; $i <= $pages; $i++): ?>
                                <a href="?page=<?= $i ?>&search=<?= urlencode($search) ?>&status=<?= urlencode($status) ?>" 
                                   class="relative inline-flex items-center px-4 py-2 border text-sm font-medium <?= $i == $page ? 'z-10 bg-indigo-50 border-indigo-500 text-indigo-600 dark:bg-indigo-900 dark:border-indigo-600 dark:text-indigo-200' : 'bg-white border-gray-300 text-gray-500 hover:bg-gray-50 dark:bg-gray-800 dark:border-gray-600 dark:text-gray-400 dark:hover:bg-gray-700' ?>">
                                   <?= $i ?>
                                </a>
                            <?php endfor; ?>
                        </nav>
                    </div>
                </div>
            </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<?php include 'footer.php'; ob_end_flush(); ?>