<?php
// Set Timezone to Dhaka, Bangladesh
date_default_timezone_set('Asia/Dhaka');

$page_title = "Dashboard";
include 'header.php'; 

// --- HELPER FUNCTIONS ---

function get_stats($pdo) {
    $stats = [];
    $check_frequency_hours = 24; 
    try {
        $stmt = $pdo->prepare("SELECT setting_value FROM system_settings WHERE setting_key = 'check_frequency_hours'");
        $stmt->execute();
        $res = $stmt->fetchColumn();
        if ($res) $check_frequency_hours = (int)$res;
    } catch (PDOException $e) {}

    $stats['total_keys'] = $pdo->query("SELECT COUNT(*) FROM license_keys")->fetchColumn();
    $stats['active_domains'] = $pdo->query("SELECT COUNT(*) FROM client_domains WHERE status = 'active'")->fetchColumn();
    $stats['revoked_count'] = $pdo->query("SELECT COUNT(DISTINCT id) FROM client_domains WHERE status = 'revoked'")->fetchColumn();
    $stats['recent_checks'] = $pdo->query("SELECT COUNT(*) FROM client_domains WHERE last_checked_at >= DATE_SUB(NOW(), INTERVAL {$check_frequency_hours} HOUR)")->fetchColumn();
    $stats['check_frequency_hours'] = $check_frequency_hours;
    
    return $stats;
}

function get_recent_domains($pdo) {
    $stmt = $pdo->prepare("
        SELECT 
            cd.domain_name, 
            cd.activated_at, 
            cd.status,
            lk.title AS license_title,
            lk.expires_at AS license_expires
        FROM client_domains cd
        LEFT JOIN license_keys lk ON cd.license_key_id = lk.id
        WHERE cd.status IN ('active', 'pending')
        ORDER BY cd.activated_at DESC 
        LIMIT 5
    ");
    $stmt->execute();
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

function time_elapsed_string($datetime, $full = false) {
    $now = new DateTime;
    $ago = new DateTime($datetime);
    $diff = $now->diff($ago);

    $diff->w = floor($diff->d / 7);
    $diff->d -= $diff->w * 7;

    $string = array('y' => 'year', 'm' => 'month', 'w' => 'week', 'd' => 'day', 'h' => 'hour', 'i' => 'minute', 's' => 'second');
    foreach ($string as $k => &$v) {
        if ($diff->$k) {
            $v = $diff->$k . ' ' . $v . ($diff->$k > 1 ? 's' : '');
        } else {
            unset($string[$k]);
        }
    }

    if (!$full) $string = array_slice($string, 0, 1);
    return $string ? implode(', ', $string) . ' ago' : 'just now';
}

// --- DATA FETCHING ---
$stats = get_stats($pdo);
$recent_domains = get_recent_domains($pdo);

// Greeting Logic
$hour = date('H');
if ($hour < 12) $greeting = "Good Morning";
elseif ($hour < 18) $greeting = "Good Afternoon";
else $greeting = "Good Evening";

// Percentages
$active_percentage = $stats['total_keys'] > 0 ? round(($stats['active_domains'] / $stats['total_keys']) * 100) : 0;
$revoked_percentage = $stats['total_keys'] > 0 ? round(($stats['revoked_count'] / $stats['total_keys']) * 100) : 0;

// Mini Chart
function render_mini_chart($percent, $colorClass) {
    return '
    <svg viewBox="0 0 36 36" class="w-16 h-16 transform -rotate-90 drop-shadow-sm">
        <path class="text-gray-100 dark:text-gray-700" d="M18 2.0845 a 15.9155 15.9155 0 0 1 0 31.831 a 15.9155 15.9155 0 0 1 0 -31.831" fill="none" stroke="currentColor" stroke-width="3" stroke-linecap="round" />
        <path class="'.$colorClass.'" stroke-dasharray="'.$percent.', 100" d="M18 2.0845 a 15.9155 15.9155 0 0 1 0 31.831 a 15.9155 15.9155 0 0 1 0 -31.831" fill="none" stroke="currentColor" stroke-width="3" stroke-linecap="round" />
    </svg>
    <div class="absolute inset-0 flex items-center justify-center flex-col">
        <span class="text-[10px] font-bold text-gray-600 dark:text-gray-300">'.$percent.'%</span>
    </div>';
}
?>

<div class="bg-gray-50/50 dark:bg-gray-900 min-h-screen py-8 font-sans transition-colors duration-200">
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        
        <!-- Welcome Header with Glass Effect -->
        <div class="mb-10 relative overflow-hidden rounded-3xl bg-gradient-to-r from-indigo-600 to-purple-700 p-8 shadow-2xl">
            <div class="absolute top-0 right-0 -mr-16 -mt-16 w-64 h-64 rounded-full bg-white opacity-10 blur-3xl"></div>
            <div class="absolute bottom-0 left-0 -ml-16 -mb-16 w-48 h-48 rounded-full bg-white opacity-10 blur-2xl"></div>
            
            <div class="relative z-10 flex flex-col md:flex-row justify-between items-center text-white">
                <div>
                    <h1 class="text-3xl md:text-4xl font-extrabold tracking-tight mb-2">
                        <?= $greeting ?>, Admin! 👋
                    </h1>
                    <p class="text-indigo-100 text-lg font-medium opacity-90">Here's what's happening with your licenses today.</p>
                </div>
                <div class="mt-6 md:mt-0 bg-white/20 backdrop-blur-md border border-white/30 rounded-2xl p-4 flex items-center shadow-lg">
                    <div class="mr-4 text-right">
                        <p class="text-xs font-bold uppercase tracking-wider text-indigo-200">Dhaka Time</p>
                        <p class="text-2xl font-mono font-bold"><?= date("h:i A") ?></p>
                    </div>
                    <div class="h-10 w-10 bg-white text-indigo-600 rounded-xl flex items-center justify-center shadow-md">
                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>
                    </div>
                </div>
            </div>
        </div>

        <!-- Stats Grid (Updated Design) -->
        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-10">
            
            <!-- Total Licenses (Blue Theme) -->
            <div class="relative bg-white dark:bg-gray-800 p-6 rounded-3xl shadow-sm hover:shadow-xl transition-all duration-300 border border-gray-100 dark:border-gray-700 group overflow-hidden">
                <div class="absolute right-0 top-0 h-24 w-24 bg-blue-50 dark:bg-blue-900/20 rounded-bl-full -mr-4 -mt-4 transition-transform group-hover:scale-110"></div>
                
                <div class="relative z-10">
                    <div class="flex items-center space-x-3 mb-4">
                        <div class="p-3 bg-blue-100 text-blue-600 rounded-2xl dark:bg-blue-900/50 dark:text-blue-400">
                            <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 7a2 2 0 012 2v2m-4 5a2 2 0 01-2-2v-2m0-6h6m-3 3v2m0 0h6m-3-6v6"></path></svg>
                        </div>
                        <p class="text-sm font-bold text-gray-500 dark:text-gray-400 uppercase tracking-wide">Total Keys</p>
                    </div>
                    <h3 class="text-4xl font-extrabold text-gray-900 dark:text-white mb-1"><?= $stats['total_keys'] ?></h3>
                    <p class="text-xs font-medium text-blue-600 dark:text-blue-400">Issued Licenses</p>
                </div>
            </div>

            <!-- Active Domains (Green Theme) -->
            <div class="relative bg-white dark:bg-gray-800 p-6 rounded-3xl shadow-sm hover:shadow-xl transition-all duration-300 border border-gray-100 dark:border-gray-700 group overflow-hidden">
                <div class="absolute right-0 top-0 h-24 w-24 bg-emerald-50 dark:bg-emerald-900/20 rounded-bl-full -mr-4 -mt-4 transition-transform group-hover:scale-110"></div>
                
                <div class="relative z-10 flex justify-between items-center">
                    <div>
                        <div class="flex items-center space-x-3 mb-4">
                            <div class="p-3 bg-emerald-100 text-emerald-600 rounded-2xl dark:bg-emerald-900/50 dark:text-emerald-400">
                                <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>
                            </div>
                            <p class="text-sm font-bold text-gray-500 dark:text-gray-400 uppercase tracking-wide">Active</p>
                        </div>
                        <h3 class="text-4xl font-extrabold text-gray-900 dark:text-white mb-1"><?= $stats['active_domains'] ?></h3>
                        <p class="text-xs font-medium text-emerald-600 dark:text-emerald-400">Online Domains</p>
                    </div>
                    <div class="relative">
                        <?= render_mini_chart($active_percentage, 'text-emerald-500') ?>
                    </div>
                </div>
            </div>

            <!-- Revoked Domains (Red Theme) -->
            <div class="relative bg-white dark:bg-gray-800 p-6 rounded-3xl shadow-sm hover:shadow-xl transition-all duration-300 border border-gray-100 dark:border-gray-700 group overflow-hidden">
                <div class="absolute right-0 top-0 h-24 w-24 bg-rose-50 dark:bg-rose-900/20 rounded-bl-full -mr-4 -mt-4 transition-transform group-hover:scale-110"></div>
                
                <div class="relative z-10 flex justify-between items-center">
                    <div>
                        <div class="flex items-center space-x-3 mb-4">
                            <div class="p-3 bg-rose-100 text-rose-600 rounded-2xl dark:bg-rose-900/50 dark:text-rose-400">
                                <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"></path></svg>
                            </div>
                            <p class="text-sm font-bold text-gray-500 dark:text-gray-400 uppercase tracking-wide">Revoked</p>
                        </div>
                        <h3 class="text-4xl font-extrabold text-gray-900 dark:text-white mb-1"><?= $stats['revoked_count'] ?></h3>
                        <p class="text-xs font-medium text-rose-600 dark:text-rose-400">Access Blocked</p>
                    </div>
                    <div class="relative">
                        <?= render_mini_chart($revoked_percentage, 'text-rose-500') ?>
                    </div>
                </div>
            </div>

            <!-- Recent Activity (Amber Theme) -->
            <div class="relative bg-white dark:bg-gray-800 p-6 rounded-3xl shadow-sm hover:shadow-xl transition-all duration-300 border border-gray-100 dark:border-gray-700 group overflow-hidden">
                <div class="absolute right-0 top-0 h-24 w-24 bg-amber-50 dark:bg-amber-900/20 rounded-bl-full -mr-4 -mt-4 transition-transform group-hover:scale-110"></div>
                
                <div class="relative z-10">
                    <div class="flex items-center space-x-3 mb-4">
                        <div class="p-3 bg-amber-100 text-amber-600 rounded-2xl dark:bg-amber-900/50 dark:text-amber-400">
                            <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 10V3L4 14h7v7l9-11h-7z"></path></svg>
                        </div>
                        <p class="text-sm font-bold text-gray-500 dark:text-gray-400 uppercase tracking-wide">Traffic</p>
                    </div>
                    <h3 class="text-4xl font-extrabold text-gray-900 dark:text-white mb-1"><?= $stats['recent_checks'] ?></h3>
                    <p class="text-xs font-medium text-amber-600 dark:text-amber-400">Hits in <?= $stats['check_frequency_hours'] ?>h</p>
                </div>
            </div>
        </div>

        <!-- Bottom Section -->
        <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
            
            <!-- Recently Activated Domains Table (License Column Removed) -->
            <div class="lg:col-span-2 bg-white dark:bg-gray-800 rounded-3xl shadow-xl border border-gray-100 dark:border-gray-700 overflow-hidden flex flex-col">
                <div class="px-8 py-6 border-b border-gray-100 dark:border-gray-700 bg-white dark:bg-gray-800 flex justify-between items-center">
                    <div>
                        <h3 class="text-xl font-bold text-gray-900 dark:text-white flex items-center">
                            <span class="bg-indigo-100 text-indigo-600 p-1.5 rounded-lg mr-3"><svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3.055 11H5a2 2 0 012 2v1a2 2 0 002 2 2 2 0 012 2v2.945M8 3.935V5.5A2.5 2.5 0 0010.5 8h.5a2 2 0 012 2 2 2 0 104 0 2 2 0 012-2h1.064M15 20.488V18a2 2 0 012-2h3.064M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg></span>
                            Recent Activations
                        </h3>
                    </div>
                    <a href="domains.php" class="group flex items-center text-sm font-semibold text-indigo-600 hover:text-indigo-800 dark:text-indigo-400 dark:hover:text-indigo-300 transition-colors">
                        View All
                        <svg class="ml-1 w-4 h-4 group-hover:translate-x-1 transition-transform" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 8l4 4m0 0l-4 4m4-4H3"></path></svg>
                    </a>
                </div>
                
                <div class="overflow-x-auto flex-grow">
                    <table class="min-w-full divide-y divide-gray-50 dark:divide-gray-700">
                        <thead class="bg-gray-50/50 dark:bg-gray-800/50">
                            <tr>
                                <th class="px-8 py-4 text-left text-xs font-bold text-gray-400 uppercase tracking-wider">Domain</th>
                                <th class="px-8 py-4 text-center text-xs font-bold text-gray-400 uppercase tracking-wider">Active Since</th>
                                <th class="px-8 py-4 text-center text-xs font-bold text-gray-400 uppercase tracking-wider hidden sm:table-cell">Valid Until</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-gray-50 dark:divide-gray-700 bg-white dark:bg-gray-800">
                            <?php if(!empty($recent_domains)): foreach ($recent_domains as $d): ?>
                            <tr class="hover:bg-gray-50/80 dark:hover:bg-gray-700/30 transition-colors duration-150">
                                <td class="px-8 py-5 whitespace-nowrap">
                                    <div class="flex items-center">
                                        <div class="flex-shrink-0 h-9 w-9 rounded-full bg-gradient-to-br from-blue-100 to-blue-50 dark:from-blue-900/30 dark:to-blue-800/30 flex items-center justify-center text-blue-600 dark:text-blue-400 shadow-sm border border-blue-100 dark:border-blue-800">
                                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 12a9 9 0 01-9 9m9-9a9 9 0 00-9-9m9 9H3m9 9a9 9 0 01-9-9m9 9c1.657 0 3-4.03 3-9s-1.343-9-3-9m0 18c-1.657 0-3-4.03-3-9s1.343-9 3-9m-9 9a9 9 0 019-9"></path></svg>
                                        </div>
                                        <div class="ml-4">
                                            <div class="text-sm font-bold text-gray-900 dark:text-white"><?= htmlspecialchars($d['domain_name']) ?></div>
                                            <div class="text-xs text-green-500 flex items-center mt-0.5">
                                                <span class="w-1.5 h-1.5 bg-green-500 rounded-full mr-1.5"></span> Online
                                            </div>
                                        </div>
                                    </div>
                                </td>
                                <td class="px-8 py-5 whitespace-nowrap text-center">
                                    <div class="text-sm font-semibold text-gray-700 dark:text-gray-300" title="<?= date("d M Y h:i A", strtotime($d['activated_at'])) ?>">
                                        <?= time_elapsed_string($d['activated_at']) ?>
                                    </div>
                                    <div class="text-xs text-gray-400"><?= date("h:i A", strtotime($d['activated_at'])) ?></div>
                                </td>
                                <td class="px-8 py-5 whitespace-nowrap text-center text-sm hidden sm:table-cell">
                                    <?php if(!empty($d['license_expires'])): 
                                        $is_exp = strtotime($d['license_expires']) < time();
                                    ?>
                                        <div class="<?= $is_exp ? 'text-red-500' : 'text-gray-700 dark:text-gray-300' ?> font-bold">
                                            <?= date("d M Y", strtotime($d['license_expires'])) ?>
                                        </div>
                                        <div class="text-xs text-gray-400"><?= date("h:i A", strtotime($d['license_expires'])) ?></div>
                                    <?php else: ?>
                                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-bold bg-indigo-100 text-indigo-700 dark:bg-indigo-900/50 dark:text-indigo-300">
                                            LIFETIME
                                        </span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                            <?php endforeach; else: ?>
                            <tr>
                                <td colspan="3" class="px-8 py-12 text-center">
                                    <div class="flex flex-col items-center">
                                        <div class="h-16 w-16 bg-gray-50 rounded-full flex items-center justify-center mb-3">
                                            <svg class="w-8 h-8 text-gray-300" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>
                                        </div>
                                        <p class="text-gray-500 font-medium">No recent activity detected.</p>
                                    </div>
                                </td>
                            </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- System Status Widget -->
            <div class="lg:col-span-1 space-y-6">
                <div class="bg-gradient-to-br from-indigo-900 to-slate-900 rounded-3xl shadow-xl p-6 text-white relative overflow-hidden">
                    <!-- Background blobs -->
                    <div class="absolute top-0 right-0 -mr-10 -mt-10 w-40 h-40 bg-indigo-500 rounded-full blur-3xl opacity-30"></div>
                    <div class="absolute bottom-0 left-0 -ml-10 -mb-10 w-40 h-40 bg-purple-500 rounded-full blur-3xl opacity-30"></div>
                    
                    <h3 class="text-lg font-bold mb-6 flex items-center relative z-10">
                        <span class="bg-white/20 p-1.5 rounded-lg mr-3 backdrop-blur-sm">
                            <svg class="w-5 h-5 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 12h14M5 12a2 2 0 01-2-2V6a2 2 0 012-2h14a2 2 0 012 2v4a2 2 0 01-2 2M5 12a2 2 0 00-2 2v4a2 2 0 002 2h14a2 2 0 002-2v-4a2 2 0 00-2-2m-2-4h.01M17 16h.01"></path></svg>
                        </span>
                        System Health
                    </h3>
                    
                    <div class="space-y-4 relative z-10">
                        <div class="bg-white/10 backdrop-blur-md rounded-xl p-4 flex items-center justify-between border border-white/10">
                            <div class="flex items-center">
                                <div class="relative flex h-3 w-3 mr-3">
                                  <span class="animate-ping absolute inline-flex h-full w-full rounded-full bg-green-400 opacity-75"></span>
                                  <span class="relative inline-flex rounded-full h-3 w-3 bg-green-500"></span>
                                </div>
                                <span class="text-sm font-medium">Database Status</span>
                            </div>
                            <span class="text-xs font-bold bg-green-500/20 text-green-300 px-2 py-1 rounded">ONLINE</span>
                        </div>

                        <div class="bg-white/10 backdrop-blur-md rounded-xl p-4 flex items-center justify-between border border-white/10">
                            <div class="flex items-center">
                                <svg class="w-4 h-4 mr-3 text-indigo-300" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>
                                <span class="text-sm font-medium">Check Interval</span>
                            </div>
                            <span class="text-sm font-mono font-bold text-indigo-300"><?= $stats['check_frequency_hours'] ?> hrs</span>
                        </div>
                        
                        <div class="bg-white/10 backdrop-blur-md rounded-xl p-4 border border-white/10">
                            <div class="flex justify-between items-end">
                                <div>
                                    <p class="text-xs text-gray-400 uppercase tracking-wide mb-1">Server Date</p>
                                    <p class="text-sm font-semibold"><?= date("l, F j, Y") ?></p>
                                </div>
                                <div class="text-right">
                                    <p class="text-2xl font-mono font-bold text-white"><?= date("h:i A") ?></p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Quick Tip -->
                <div class="bg-indigo-50 dark:bg-indigo-900/20 rounded-2xl p-5 border border-indigo-100 dark:border-indigo-800">
                    <div class="flex">
                        <div class="flex-shrink-0">
                            <svg class="h-5 w-5 text-indigo-400" fill="currentColor" viewBox="0 0 20 20"><path fill-rule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clip-rule="evenodd"></path></svg>
                        </div>
                        <div class="ml-3">
                            <h3 class="text-sm font-bold text-indigo-800 dark:text-indigo-300">Quick Tip</h3>
                            <div class="mt-2 text-sm text-indigo-700 dark:text-indigo-400">
                                <p>You can revoke access instantly from the Domains page if you notice suspicious activity.</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

        </div>
    </div>
</div>

<?php include 'footer.php'; ob_end_flush(); ?>