<?php
// Start output buffering (for safety)
ob_start();

// --- API Configuration and Setup ---
// 1. Include the configuration file to initialize $pdo and utility functions
// NOTE: Assuming config.php is located two directories up from api/v1/
// Adjust this path if your folder structure is different
if (file_exists('../../config.php')) {
    include '../../config.php';
}

// **$pdo is now initialized globally from config.php**

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *'); // IMPORTANT: Limit this to only known client domains in production!

// Function to send JSON response and exit
function respond($status, $message, $data = []) {
    // Determine the HTTP response code
    $http_code = ($status === 'success') ? 200 : 400;
    
    // For critical errors (like DB not connected), use 500
    if ($status === 'error' && strpos($message, 'internal database error') !== false) {
        $http_code = 500;
    }
    
    http_response_code($http_code);
    $response = ['status' => $status, 'message' => $message];
    if (!empty($data)) {
        $response = array_merge($response, $data);
    }
    echo json_encode($response);
    ob_end_flush();
    exit;
}

// Function to sanitize domain name
function sanitize_domain($domain) {
    // Remove protocol and trailing slash, then lowercase
    $domain = preg_replace('/^https?:\/\//', '', $domain);
    $domain = rtrim($domain, '/');
    return strtolower(trim($domain));
}

// Function to validate key existence and status
function validate_license($pdo, $api_key, $secret_key) {
    // We assume execute_query is defined either in config.php or a shared utility file
    if (!function_exists('execute_query')) {
        return ['valid' => false, 'message' => 'Internal error: execute_query function missing.'];
    }
    
    // UPDATED SQL: Added module columns to the select list
    $sql = "
        SELECT id, is_active, title, secret_key, module_ecommerce, module_filesystem, module_membership
        FROM license_keys
        WHERE api_key = :api_key
    ";
    $license = execute_query($pdo, $sql, [':api_key' => $api_key], false);

    if (!$license) {
        return ['valid' => false, 'message' => 'License Key not found.'];
    }
    
    // CRITICAL SECURITY CHECK: Secret Key Verification
    if ($license['secret_key'] !== $secret_key) {
        return ['valid' => false, 'message' => 'Invalid license combination.'];
    }

    if ($license['is_active'] == 0) {
        return ['valid' => false, 'message' => 'License has been revoked by the administrator.'];
    }
    
    // License is valid - preparing modules array
    // We cast to int/bool to ensure clean JSON output
    $modules = [
        'module_ecommerce'  => (int)$license['module_ecommerce'],
        'module_filesystem' => (int)$license['module_filesystem'],
        'module_membership' => (int)$license['module_membership']
    ];
    
    return [
        'valid' => true, 
        'license_id' => $license['id'], 
        'title' => $license['title'],
        'modules' => $modules // Passing modules back
    ];
}

// Function to fetch or update domain status (fallback definition)
if (!function_exists('execute_query')) {
    function execute_query($pdo, $sql, $params = [], $fetch_all = false) {
        if (!isset($pdo) || !$pdo instanceof PDO) { return $fetch_all ? [] : 0; }
        $normalized_sql = trim(strtoupper($sql));
        try {
            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);
            $is_count_query = (substr($normalized_sql, 0, 12) === 'SELECT COUNT'); 
            if ($fetch_all) { return $stmt->fetchAll(\PDO::FETCH_ASSOC); }
            if ($is_count_query) { return (int)$stmt->fetchColumn(); }
            return $stmt->fetch(\PDO::FETCH_ASSOC); 
        } catch (\PDOException $e) {
            error_log("API Database Error: " . $e->getMessage());
            return $is_count_query ? 0 : ($fetch_all ? [] : null); 
        }
    }
}

// --- Main API Routing ---

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    respond('error', 'Only POST requests are accepted.');
}

// Get the POST data
$input = json_decode(file_get_contents('php://input'), true);

$action = $input['action'] ?? null;
$api_key = $input['api_key'] ?? null;
$secret_key = $input['secret_key'] ?? null;
$raw_domain = $input['domain'] ?? null;
$domain = sanitize_domain($raw_domain);
$current_ip = $_SERVER['REMOTE_ADDR']; 

if (empty($action) || empty($api_key) || empty($secret_key) || empty($domain)) {
    respond('error', 'Missing required parameters: action, api_key, secret_key, or domain.');
}

// CRITICAL CHECK: Ensure $pdo is connected
global $pdo;
if (!isset($pdo) || !$pdo instanceof PDO) {
    // Attempting to reconnect if global pdo failed (optional safety net)
    if (defined('DB_SERVER') && defined('DB_NAME')) {
        try {
             $dsn = "mysql:host=" . DB_SERVER . ";dbname=" . DB_NAME . ";charset=utf8mb4";
             $pdo = new PDO($dsn, DB_USERNAME, DB_PASSWORD, [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]);
        } catch (PDOException $e) {
             respond('error', 'API is offline due to internal database error.', 500);
        }
    } else {
        respond('error', 'API is offline due to internal database error.', 500);
    }
}

// 1. Validate License Keys for all actions
$license_check = validate_license($pdo, $api_key, $secret_key);

if (!$license_check['valid']) {
    respond('error', $license_check['message']);
}

$license_id = $license_check['license_id'];
$modules = $license_check['modules'] ?? []; // Extract modules

// --- Handle Specific Actions ---

if ($action === 'activate') {
    
    // Check if domain is already linked to this or another license
    $domain_check_sql = "
        SELECT id, license_key_id, status
        FROM client_domains
        WHERE domain_name = :domain
    ";
    $existing_domain = execute_query($pdo, $domain_check_sql, [':domain' => $domain], false);

    if ($existing_domain) {
        if ((int)$existing_domain['license_key_id'] === (int)$license_id && $existing_domain['status'] === 'active') {
             // Domain match & active: Return Success + Modules
             respond('success', 'Domain is already activated with this key.', [
                 'domain_status' => 'active',
                 'modules' => $modules
             ]);
        } elseif ((int)$existing_domain['license_key_id'] !== (int)$license_id) {
             respond('error', 'Domain is already linked to a different license key.', ['domain_status' => $existing_domain['status']]);
        } elseif ($existing_domain['status'] === 'revoked' || $existing_domain['status'] === 'expired') {
            // Re-activate
            $update_sql = "
                UPDATE client_domains
                SET status = 'active', last_checked_at = NOW(), last_check_ip = :ip
                WHERE id = :id AND license_key_id = :license_id
            ";
            $stmt = $pdo->prepare($update_sql);
            $update_success = $stmt->execute([
                ':id' => $existing_domain['id'],
                ':license_id' => $license_id,
                ':ip' => $current_ip
            ]);
            
            if ($update_success) {
                respond('success', 'Domain reactivated successfully.', [
                    'domain_status' => 'active',
                    'modules' => $modules
                ]);
            } else {
                respond('error', 'Failed to reactivate existing domain entry.');
            }
        } else {
            // Pending or other states
            respond('success', 'Domain activated successfully.', [
                'domain_status' => 'active',
                'modules' => $modules
            ]);
        }
    } else {
        // Insert new entry
        $insert_sql = "
            INSERT INTO client_domains (license_key_id, domain_name, status, activated_at, last_check_ip, last_checked_at)
            VALUES (:license_id, :domain, 'active', NOW(), :ip, NOW())
        ";
        try {
            $stmt = $pdo->prepare($insert_sql);
            $stmt->execute([
                ':license_id' => $license_id,
                ':domain' => $domain,
                ':ip' => $current_ip
            ]);
            respond('success', 'License activated and domain registered.', [
                'domain_status' => 'active',
                'modules' => $modules
            ]);
        } catch (\PDOException $e) {
            error_log("Domain Insert Error: " . $e->getMessage());
            respond('error', 'Failed to register new domain entry.');
        }
    }

} elseif ($action === 'status_check') {
    
    $domain_status_sql = "
        SELECT status, expires_at
        FROM client_domains
        WHERE domain_name = :domain AND license_key_id = :license_id
    ";
    $domain_check = execute_query($pdo, $domain_status_sql, [
        ':domain' => $domain,
        ':license_id' => $license_id
    ], false);
    
    // Update last check details
    try {
        $update_check_sql = "UPDATE client_domains SET last_checked_at = NOW(), last_check_ip = :ip WHERE domain_name = :domain AND license_key_id = :license_id";
        $stmt = $pdo->prepare($update_check_sql);
        $stmt->execute([':domain' => $domain, ':license_id' => $license_id, ':ip' => $current_ip]);
    } catch (\PDOException $e) {
         error_log("Failed to update check time: " . $e->getMessage());
    }

    if (!$domain_check) {
        respond('error', 'Domain is not registered under this license key. Please activate first.');
    }
    
    if ($domain_check['status'] === 'active') {
        respond('success', 'License is active for this domain.', [
            'domain_status' => 'active',
            'expires_at' => $domain_check['expires_at'],
            'modules' => $modules // IMPORTANT: Sending modules data
        ]);
    } else {
        respond('error', 'License status is ' . $domain_check['status'], [
            'domain_status' => $domain_check['status']
        ]);
    }

} else {
    respond('error', 'Invalid action specified.');
}
?>